/* VHostConf.java -- Holds information for hosts running off a single port
   Copyright (C) 1999 Darrick Wong
*/
import java.util.*;
import java.io.*;

public class VHostConf {
  //Statistical Stuff
    //keep a track of how many requests have been sent to this virtual host.
    private httpdconf m_hcHttpdConf;
    private long m_lNumHostRequests;
    //buffer size for file transfer
    private int m_iBufferSize;

  //Browser appearance
    //can list folders
    private boolean m_bListFolders;
    //can show icons in those listings
    private boolean m_bShowIcons;
    //uses tables for better-looking tables
    private boolean m_bListFolderUsingTables;
    //allow browser to cache material?
    private boolean m_bBrowserCachingEnabled;
    //use javascript effects in directory listings?
    private boolean m_bUseJavascriptInTableListings;
    //use combo box for directory listings? (JAVASCRIPT MUST BE ON!)
    private boolean m_bUseComboBoxForDirectoryNavigation;

  //Operational stuff...logging
    private PrintStream m_psNormalLog, m_psErrorLog;
    //file system case sensitive?
    private boolean m_bFileSystemCaseSensitive;
    //looks for default files?  (index.html, index.htm, default.html, default.htm)
    private boolean m_bLookForDefaultFiles;
    //VFS folder...
    private String  m_sVFSRoot;
    //admin's email address (for error messages)
    private String  m_sWebAdminEmail;
    //where to serve pages out of
    private String  m_sDocumentRoot;
    //CGI executables' home (can be document root, although that's a bit dangerous!)
    private String  m_sCgiRoot;
    //time that we keep forwarding data from CGIs after the program has closed.
    private long m_lCgiOutputAfterDeathTimeout;
    //mime database...
    private MimeDatabase  m_mdMimeDatabase;
    private String  m_sMimeDatabaseFile;
    //host name
    private String  m_sHostName;

    private Vector m_vDefaultFiles;

  //Accessors
    public int getBufferSize()                  {return m_iBufferSize;}

    public long getNumHostRequests()            {return m_lNumHostRequests;}
    public long getCgiOutputAfterDeathTimeout() {return m_lCgiOutputAfterDeathTimeout;}

    public boolean canShowIcons()               {return m_bShowIcons;}
    public boolean canListFolderUsingTables()   {return m_bListFolderUsingTables;}
    public boolean canListFolders()             {return m_bListFolders;}
    public boolean canLookForDefaultFiles()     {return m_bLookForDefaultFiles;}
    public boolean isBrowserCachingEnabled()    {return m_bBrowserCachingEnabled;}
    public boolean isFileSystemCaseSensitive()  {return m_bFileSystemCaseSensitive;}
    public boolean canUseJavascriptInTableListings()    {return m_bUseJavascriptInTableListings;}
    public boolean canUseComboBoxForDirectoryNavigation()   {return m_bUseComboBoxForDirectoryNavigation;}

    public String getVFSRoot()                  {return m_sVFSRoot;}
    public String getWebAdminEmail()            {return m_sWebAdminEmail;}
    public String getDocumentRoot()             {return m_sDocumentRoot;}
    public String getHostName()                 {return m_sHostName.toLowerCase();}
    public String getCgiRoot()                  {return (m_sCgiRoot==null?getDocumentRoot()+"cgi-bin"+File.separatorChar:m_sCgiRoot);}

    public MimeDatabase getMimeDatabase()       {return m_mdMimeDatabase;}
    public String getMimeDatabaseFile()         {return m_sMimeDatabaseFile;}

    public PrintStream getNormalStream()        {return m_psNormalLog;}
    public PrintStream getErrorStream()         {return m_psErrorLog;}
    public Vector getDefaultFiles()             {return m_vDefaultFiles;}

  //Setters...please find a real word...
    public void setBufferSize(int d)                    {m_iBufferSize = d;}

    public void setShowIcons(boolean b)                 {m_bShowIcons = b;}
    public void setListFolderUsingTables(boolean b)     {m_bListFolderUsingTables = b;}
    public void setListFolders(boolean b)               {m_bListFolders = b;}
    public void setLookForDefaultFiles(boolean b)       {m_bLookForDefaultFiles = b;}
    public void setBrowserCachingEnabled(boolean b)     {m_bBrowserCachingEnabled = b;}
    public void setFileSystemCaseSensitive(boolean b)   {m_bFileSystemCaseSensitive = b;}
    public void setUseJavascriptInTableListings(boolean b)  {m_bUseJavascriptInTableListings = b;}
    public void setUseComboBoxForDirectoryNavigation(boolean b) {m_bUseComboBoxForDirectoryNavigation = b;}

    public void setDefaultFiles(Vector v)               {m_vDefaultFiles = v;}

    public void setVFSRoot(String s)                    {m_sVFSRoot = s;}
    public void setWebAdminEmail(String s)              {m_sWebAdminEmail = s;}
    public void setDocumentRoot(String s)               {m_sDocumentRoot = s;}
    public void setHostName(String s)                   {m_sHostName = s.toLowerCase();}
    public void setCgiRoot(String s)                    {m_sCgiRoot = s;}

    public void setMimeDatabase(MimeDatabase p)         {m_mdMimeDatabase = p;}
    public void setMimeDatabaseFile(String s)           {m_sMimeDatabaseFile = s;setMimeDatabase(new MimeDatabase(this, m_hcHttpdConf));}

    public void incNumHostRequests()                    {m_lNumHostRequests++;}
    public void decNumHostRequests()                    {m_lNumHostRequests--;}  //umm...why?

    public void setNormalStream(PrintStream s)          {m_psNormalLog = s;}
    public void setErrorStream(PrintStream s)           {m_psErrorLog = s;}

    public void setCgiOutputAfterDeathTimeout(long l)   {m_lCgiOutputAfterDeathTimeout = l;}

  //constructor
    public VHostConf(httpdconf hc, String hostname) {
        if(hostname == "") {
            try {
                hostname = java.net.InetAddress.getLocalHost().getHostName().toLowerCase();
            } catch (java.net.UnknownHostException e) {
                hostname = "localhost";
            }
        }

        m_hcHttpdConf = hc;        

        if(m_hcHttpdConf.getDebugLevel() > 0)
            m_hcHttpdConf.print(Thread.currentThread().getName(), "Initializing vhost "+hostname+":");

        m_lNumHostRequests = 0;
        m_bListFolders = true;
        m_bShowIcons = true;
        m_bListFolderUsingTables = true;
        m_bBrowserCachingEnabled = true;
        m_bFileSystemCaseSensitive = false;
        m_bLookForDefaultFiles = true;
        m_sVFSRoot = "./VFS/";
        m_sWebAdminEmail = "webmaster@localhost";
        m_sDocumentRoot = "./htdocs/";
        m_sCgiRoot = null;
        m_sHostName = hostname;
        m_psNormalLog = System.out;
        m_psErrorLog = System.err;
        m_iBufferSize = 131072;
        m_bUseJavascriptInTableListings = true;
        m_bUseComboBoxForDirectoryNavigation = true;
        m_lCgiOutputAfterDeathTimeout = 1000;

        if(m_hcHttpdConf.getDebugLevel() > 0)
            m_hcHttpdConf.println(" vhost");

        m_sMimeDatabaseFile = "./mimetype.txt";
        m_mdMimeDatabase = new MimeDatabase(this, m_hcHttpdConf); //WARNING: this line MUST come after the datbase file initializer

        m_vDefaultFiles = new Vector();
        m_vDefaultFiles.addElement("index.html");
        m_vDefaultFiles.addElement("index.htm");
        m_vDefaultFiles.addElement("default.html");
        m_vDefaultFiles.addElement("default.htm");                        
    }

    public void dumpConfig(StringBuffer result) {
        result.append("<H2>Virtual Host named \""+m_sHostName.toLowerCase()+"\":</H2>\r\n");
        result.append("<UL>\r\n");

        result.append("<LI>Allows browser to cache data? "+m_bBrowserCachingEnabled+"</LI>\r\n");
        result.append("<LI>Can send directory listings? "+m_bListFolders+"</LI>\r\n");
        result.append("<LI>File transfer buffer size = "+m_iBufferSize+" bytes</LI>\r\n");
        result.append("<LI>Folder to serve CGIs out of = "+getCgiRoot()+"</LI>\r\n");
        result.append("<LI>Folder to serve files out of = "+m_sDocumentRoot+"</LI>\r\n");
        result.append("<LI>Length of time to forward CGI output after program death = "+m_lCgiOutputAfterDeathTimeout+"ms</LI>\r\n");
        result.append("<LI>Looks for default files? "+m_bLookForDefaultFiles+"</LI>\r\n");
        result.append("<LI>MIME database file = "+m_sMimeDatabaseFile+"</LI>\r\n");
        result.append("<LI>Requests to this host = "+m_lNumHostRequests+"</LI>\r\n");
        result.append("<LI>Shows icons in directory listings? "+m_bShowIcons+"</LI>\r\n");
        result.append("<LI>Treats filesystem as case sensitive? "+m_bFileSystemCaseSensitive+"</LI>\r\n");
        result.append("<LI>Use a ComboBox For Directory Navigation? "+m_bUseComboBoxForDirectoryNavigation+"</LI>\r\n");
        result.append("<LI>Use JavaScript in tabled directory listings? "+m_bUseJavascriptInTableListings+"</LI>\r\n");
        result.append("<LI>Uses tables for directory listings? "+m_bListFolderUsingTables+"</LI>\r\n");
        result.append("<LI>VFS root folder = "+m_sVFSRoot+"</LI>\r\n");
        result.append("<LI>Web admin's email = "+m_sWebAdminEmail+"</LI>\r\n");
        result.append("<LI>Default files = ");
        Enumeration enu = m_vDefaultFiles.elements();
        while(enu.hasMoreElements()) {
            result.append(enu.nextElement());
            if(enu.hasMoreElements()) {result.append(", ");}
        }
        result.append("</LI>\r\n");
        result.append("</UL>\r\n");
    }

    public static VHostConf generateConfigFromStrings(BufferedReader br, SuperInt line, httpdconf hc) throws Exception {
        String input, key, value;
        int e, s;
        VHostConf vhc = new VHostConf(hc, "");

        while( (input=br.readLine()) != null) {
            input = input.trim();
            line.increment();

            if(input.startsWith("//") || input.startsWith(";") || input.startsWith("#") || input.equals("")) continue;

            if(input.equals("}"))
                return vhc;

            if( (s = input.indexOf(';')) == -1) throw new Exception("Parse error, line "+line+": Semicolon not found.");
            if( (e = input.indexOf('=')) == -1) throw new Exception("Parse error, line "+line+": Equals sign not found.");
            key = input.substring(0, e).toLowerCase();
            value = input.substring(e+1, s);

            if(key.equals("buffersize")) {
                vhc.setBufferSize(Integer.parseInt(value));

            } else if(key.equals("showicons")) {
                vhc.setShowIcons((value.toLowerCase().equals("true")?true:false));
            } else if(key.equals("listfolderusingtables")) {
                vhc.setListFolderUsingTables((value.toLowerCase().equals("true")?true:false));
            } else if(key.equals("listfolders")) {
                vhc.setListFolders((value.toLowerCase().equals("true")?true:false));
            } else if(key.equals("lookfordefaultfiles")) {
                vhc.setLookForDefaultFiles((value.toLowerCase().equals("true")?true:false));
            } else if(key.equals("browsercachingenabled")) {
                vhc.setBrowserCachingEnabled((value.toLowerCase().equals("true")?true:false));
            } else if(key.equals("filesystemcasesensitive")) {
                vhc.setFileSystemCaseSensitive((value.toLowerCase().equals("true")?true:false));
            } else if(key.equals("usejavascriptintablelistings")) {
                vhc.setUseJavascriptInTableListings((value.toLowerCase().equals("true")?true:false));
            } else if(key.equals("usecomboboxfordirectorynavigation")) {
                vhc.setUseComboBoxForDirectoryNavigation((value.toLowerCase().equals("true")?true:false));
            } else if(key.equals("cgioutputafterdeathtimeout")) {
                vhc.setCgiOutputAfterDeathTimeout(Long.parseLong(value));

            } else if(key.equals("mimedatabase")) {
                vhc.setMimeDatabaseFile(value);
            } else if(key.equals("vfsroot")) {
                vhc.setVFSRoot(value);
            } else if(key.equals("webadminemail")) {
                vhc.setWebAdminEmail(value);
            } else if(key.equals("documentroot")) {
                vhc.setDocumentRoot(value);
            } else if(key.equals("cgiroot")) {
                vhc.setCgiRoot(value);
            } else if(key.equals("hostname")) {
                vhc.setHostName(value);
            } else if(key.equals("defaultfiles")) {
                //parse the value; files are separated by commas
                StringTokenizer st = new StringTokenizer(value, ",", false);
                ArrayableVector v = new ArrayableVector();
                while(st.hasMoreTokens())
                    v.addElement(st.nextToken());
                vhc.setDefaultFiles(v);

            } else {
                throw new Exception("Parse error, line "+line+": Unrecognized directive \""+key+"\".");
            }

//            System.out.println("VHostConf: \""+key+"\"=\""+value+"\"");
        }
        //If we get here, we haven't gotten to a close brace--an error has occured.
        throw new Exception("Parse error, line "+line+": Closing brace not found.");
    }
}
