/* RequestHandler.java -- handle HTTP requests...
   Copyright (C) 1999 Darrick Wong

   NOTE: We still haven't got a solution for the sleep() and notify() problem.

   Somewhat Likely To Implement:
   =============================
   Error 301 Moved Permanently
   Error 302 Moved Temporarily
   Error 410 Gone
   Error 408 Timeout -- Must implement the timeout stuff
   Keepalive connections (will have to change to HTTP/1.1 for that...)

   Likely to Implement:
   ====================
   Error 415 Media not supported -- add option to refuse transfer for file types not in MIME database.
*/
import java.net.*;
import java.io.*;
import java.util.*;
import java.text.*;

public class RequestHandler extends Thread {
    private ThreadPool m_tpHolder;
    private ServerConf m_scServerConf;
    private httpdconf m_hcHttpdConf;
    private boolean m_bHasData;
    private boolean m_bAddedToThreadQueue;

    //per-request data
    private Socket m_Socket; //do not nullify this
    private VHostConf m_vhcHostConf;
    private String m_sCommand;
    private String m_sFirstLine;
    private String m_sRequestVersion;

    private String m_sResponse;
    private StringBuffer m_sbHeaders;
/*    private StringBuffer m_sbTextContents;
    private byte[] m_baOutput;
    private String m_sContentType;
    private String m_sLocalFile;*/

    private InetAddress m_iaThisMachine, m_iaClient;
    private InputStream m_isRequestStream;
    private OutputStream m_osResponseStream;
    private DataOutputStream m_dosResponseStream;
    private DataInputStream m_disRequestStream;
    private Properties m_prHeaders;//, m_prArguments;
    private boolean m_bGotServer;

    public static final int iMinMajorVersion = 1;
    public static final int iMaxMajorVersion = 1;
    public static final int iMinMinorVersion = 0;
    public static final int iMaxMinorVersion = 1;

    private void resetData() {
        m_sCommand = null;
        m_sRequestVersion = null;

        m_sResponse = null;
        m_sFirstLine = null;
        m_sbHeaders = new StringBuffer();
/*        m_sbTextContents = null;
        m_baOutput = null;
        m_sLocalFile = null;*/
        m_iaClient = null;
//        m_sContentType = null;
        m_prHeaders.clear();
        m_isRequestStream = null;
        m_osResponseStream = null;
        m_vhcHostConf = null;
        m_bGotServer = false;
//        m_prArguments.clear();
        m_prHeaders.clear();
    }

    public static final String crlf = "\r\n";

    public boolean isAddedToThreadQueue() {return m_bAddedToThreadQueue;}
    public void setAddedToThreadQueue(boolean b) {m_bAddedToThreadQueue = b;}

    public String getCommand() {return m_sCommand;}
    public String getFirstLine() {return m_sFirstLine;}

    //constructor
    private RequestHandler() {}

    public RequestHandler(ThreadPool holder, httpdconf hc) {
        super();
        m_hcHttpdConf = hc;
        if(m_hcHttpdConf.getDebugLevel() > 4) {
            m_hcHttpdConf.println(getName(), "New request handler created.");
        }

        m_tpHolder = holder;
        m_bHasData = false;
        m_bAddedToThreadQueue = false;
        m_prHeaders = new Properties();

        resetData();        

        setDaemon(true);
        try {
            m_iaThisMachine = InetAddress.getLocalHost();
        } catch (UnknownHostException e) {
            dumpException(e);
        }
        start();
    }

    public void sendRequest(ServerConf sc, Socket s) throws IOException {
        if(m_bHasData) throw new IOException(getName()+": Thread already in use.");
        m_scServerConf = sc;
        m_Socket = s;
        m_bHasData = true;
        m_bGotServer = true;
        m_iaClient = m_Socket.getInetAddress();
        printDebugMessage(4, "Request received by RequestHandler...");
    }

    public void run() {
        while(true) {
            if(isInterrupted()) {
                m_hcHttpdConf.println(getName(), "Thread destroyed.");
                if(m_bAddedToThreadQueue) {
                    m_hcHttpdConf.println(getName(), "AAAAAAAAAAA!  ACCIDENTAL THREAD DEATH!");
                }
                return;
            } else if(m_bHasData) {
                if(m_scServerConf.getDebugLevel() > 4) {
                    m_scServerConf.println(getName(), "Handling request...");
                }
                handleRequest();
            } else {
                try {
                    sleep(100);  //um....
                } catch (Exception e) {
                    dumpException(e); continue;
                }
            }
        }
    }

    private void handleRequest() {
        InputStream is = null;
        OutputStream os = null;
//        BufferedReader br = null;
        try {
            /*  HANDLING REQUESTS
                =================

                This is what various browsers send us.  Headers have been sorted alphabetically.
              IE 5.0A (5.00.2314)
                GET /output.txt HTTP/1.1
                Accept-Encoding: gzip, deflate
                Accept-Language: en-us
                Accept: * / *
                Connection: Keep-Alive
                Host: this
                If-Modified-Since: Sat, 12 Jun 1999 17:18:32 GMT; length=915
                User-Agent: Mozilla/4.0 (compatible; MSIE 5.0; Windows 98; DigExt)

              NS 4.61
                GET /output.txt HTTP/1.0
                Accept-Charset: iso-8859-1,*,utf-8
                Accept-Encoding: gzip
                Accept-Language: en
                Accept: image/gif, image/x-xbitmap, image/jpeg, image/pjpeg, image/png, * / *
                Connection: Keep-Alive
                Host: this
                If-Modified-Since: Sat, 12 Jun 1999 17:19:42 GMT; length=1381
                Pragma: no-cache
                User-Agent: Mozilla/4.61 [en] (Win98; I)

              NS 3.04
                GET /output.txt HTTP/1.0
                Accept: image/gif, image/x-xbitmap, image/jpeg, image/pjpeg, * / *
                Connection: Keep-Alive
                Host: this
                If-Modified-Since: Saturday, 12-Jun-99 17:17:26 GMT; length=3703
                Pragma: no-cache
                User-Agent: Mozilla/3.04 (Win95; I)

              NS 1.22
                GET /output.txt HTTP/1.0
                Accept: * / *
                Accept: image/gif
                Accept: image/jpeg
                Accept: image/x-xbitmap
                If-Modified-Since: Saturday, 12-Jun-99 17:15:04 GMT
                Pragma: no-cache
                User-Agent: Mozilla/1.22KIT (Windows; I; 16bit)

                What we do here is set up the environment for the specific handler.
                We initialize anything and everything that is COMMON to EVERY request--
                read the headers, set return code, set return headers, write data
                --but NOTHING that is specific to the request.  We don't attempt to
                interpret any input, aside from the command and HTTP version in the
                first line.

                Then call the appropriate handler.
            */

            //open streams
            try {
                m_isRequestStream = m_Socket.getInputStream();
                m_osResponseStream = m_Socket.getOutputStream();
                m_disRequestStream = new DataInputStream(m_isRequestStream);
//                br = new BufferedReader(new InputStreamReader(m_isRequestStream));
                m_dosResponseStream = new DataOutputStream(m_osResponseStream);
            } catch (IOException e) {
                dumpException(e);
                return; //error! error!
            }

            //dump request reception time, socket info time
            if(m_scServerConf.getDebugLevel() > 0) {
                m_scServerConf.println(getName(), "Request received at: "+
                    SuperFile.getLastModified(System.currentTimeMillis())+"."
                );
            }
            if(m_scServerConf.getDebugLevel() > 1) {m_scServerConf.println(getName(), "Client socket info: "+m_Socket);}

            //read command
            String input;

            /*  DETECTING DEAD SOCKETS
                ======================

                We have a problem--if browsers/hackers open up 300 connections
                and proceed to sit on them, we get a bunch of dead threads.  However,
                if they just sit, br.readLine() returns NULL.  So, here's the plan to
                counter such a problem:

                1) Note the current time.
                2) while input is null, check current time against the noted time.
                    if they're greater than our timeout, then kill the connection.
            */

            //Grab command
            long noted = System.currentTimeMillis();
            long f;

            while( (input = m_disRequestStream.readLine()) == null) {
//            while( (input = br.readLine()) == null) {
                f = System.currentTimeMillis() - noted;
                if( (f) > m_scServerConf.getTimeoutMillis()) {
                    if(m_scServerConf.getDebugLevel() > 1) {
                        m_scServerConf.println(getName(), "Timed out.");
                    }
                    return;
                }
                yield();
            }
            if(m_scServerConf.getDebugLevel() > 1) {m_scServerConf.println(getName(), "Command: "+input);}

            //extract the command
            m_sFirstLine = input;
            int iTemp = input.indexOf(' ');
            m_sCommand = input.substring(0, iTemp).toLowerCase();

            //then get version
            iTemp = input.toLowerCase().lastIndexOf("http/");
            if(iTemp >= 0) m_sRequestVersion = input.substring(iTemp+5);

            if(m_scServerConf.getDebugLevel() > 1) {
                m_scServerConf.println(getName(), "HTTP Version: "+m_sRequestVersion);
            }

            //if wrong version, abort
            if(!isSupportedHttpVersion()) {
                signalError(505);
                return;
            }
            if(m_scServerConf.getDebugLevel() > 1) {m_scServerConf.println(getName(), "Correct HTTP version.");}

            /*  Now we're ready to begin reading headers.
                Because incoming streams might be slow (read: coming from
                Stanek's 2400 baud modem), we have to implement our little
                timeout loop.  Unfortunately, this applies to header reading,
                too...the problem is in the while loop's conditions.
                I suppose we could do something like read, then engage a do-while
                loop...
            */
            m_prHeaders.clear();
            int i = 1; String key, value;

            //read data...
            noted = System.currentTimeMillis();
//            while( (input = br.readLine().trim()) == null) {
            while( (input = m_disRequestStream.readLine().trim()) == null) {
                if( (System.currentTimeMillis() - noted) > m_scServerConf.getTimeoutMillis()) return;
                yield();
            }

            //read stuff into Property list
            while ( !input.equals("") ) {
                if(m_scServerConf.getDebugLevel() > 2) {m_scServerConf.println(getName(), "Header line "+(i++)+": "+input);}

                iTemp = input.indexOf(':');
                try {
                    key = input.substring(0, iTemp);
                    value = input.substring(iTemp+2);

                    if((!key.trim().equals("")) && (!(value.trim().equals(""))))
                        m_prHeaders.put(key, value);
                } catch (Exception e) {break;}

                //addition: read the next line or server locks up...
                noted = System.currentTimeMillis();

                while( (input = m_disRequestStream.readLine().trim()) == null) {
//                while( (input = br.readLine().trim()) == null) {
                    if( (System.currentTimeMillis() - noted) > m_scServerConf.getTimeoutMillis()) return;
                    yield();
                }
            }

            //Figure out what Virtual Host we should use.  Technically, this violates
            //our policy of NOT interpreting data, but this is common to all of
            //the requests.
            //We've parsed the headers.  Now we can look for the Hosts.

            //first, set the default host as the zeroth in our array.
            Vector hosts = m_scServerConf.getVirtualHosts();
            Enumeration e = hosts.elements();
            try {
                m_vhcHostConf = (VHostConf)hosts.elementAt(0);
            } catch (ArrayIndexOutOfBoundsException eep) {
                if(m_scServerConf.getDebugLevel() > 0) {
                    m_scServerConf.println(getName(), "Error!  No Virtual Hosts have been set up!");
                }
                signalError(503);
                return;
            }

            //grab the Host header...
            input = m_prHeaders.getProperty("Host");

            //if it's null, (non-existent), skip this step...otherwise search our list.
            if(input != null) {
                input = input.toLowerCase();

                int b = input.indexOf(':'); //remove port number...
                if(b >= 0) {
                    input = input.substring(0, b);
                }

                VHostConf hc;

                while(e.hasMoreElements()) {
                    hc = (VHostConf)e.nextElement();
                    if(hc.getHostName().equals(input)) {m_vhcHostConf = hc; break;}
                }
            }

            printDebugMessage(2, "Using virtual host: "+m_vhcHostConf.getHostName());
            m_vhcHostConf.incNumHostRequests();

            //NOW handle commands.
            if((m_sCommand.equals("get") || m_sCommand.equals("head"))) {
                printDebugMessage(1, "Entering GetAndHeadHandler...");
                GetAndHeadHandler.handleCommand(this);
            }

            printDebugMessage(1, "Done handling command \""+m_sCommand+"\"");

        } catch (Throwable t) {
            dumpException(t);
//            m_scServerConf.errln(getName(), "Ooh!  Uncaught exception: "+t.getMessage());
        } finally {
            try {
                m_Socket.close();
            } catch (IOException e) {
                dumpException(e);
            }
            if(m_scServerConf.getDebugLevel() > 4) {
                m_scServerConf.println(getName(), "Finished Request");
            }
            m_bHasData = false;
            resetData();
            m_Socket = null;
            m_scServerConf = null;
            m_tpHolder.returnRequestHandler(this);
        }
    }


    /*  WRITING OUTPUT
        ==============

        First, write the status code (HTTP/1.0 200 OK)
        Then, figure out the length of the output.
        set a header called Content-Length
        set a header called Content-Type
        Write the headers
        Write the byte output.

        Initially, I thought of creating 3 buffers (response, headers,
        contents) but then realized that the memory cost would be
        astronomical.  So, I changed things so that each function that
        sends responses is responsible for writing all 3.

        Look throughout the source for "*** WRITE DATA ***".
    */

    public void signalError(int error) {signalError(error, -1, "");}
    public void signalError(int error, String data) {signalError(error, -1, data);}
    public void signalError(int error, int subcode, String data) {
        String title, descr, body;
        if(m_scServerConf.getDebugLevel() > 3) {m_scServerConf.println(getName(), "Error: "+error+"."+subcode);}

        setStatusCode(error);
        switch(error) {
            case -1: {
                title = "Invalid error code";
                descr = "Error: Invalid error code.";
                body  = "You have selected an invalid error code.";
                break;
            }
            case 200: {return;} //no error
            case 204: {
                title = "204 No Content";
                descr = "Error 204: No Content.";
                body  = "There are no data to send back.";
                break;
            }
            case 304: {return;} //no error
            case 400: {
                title = "400 Malformed Request";
                descr = "Error 400: Bad request.";
                body  = "Your web browser sent \""+m_sCommand+"\" as command; "+
                    "requests should be formed [GET|HEAD] filename HTTP/1.0.";
                break;
            }
            case 403: {
                switch(subcode) {
                    case 1: {
                        title = "403.1 Relative Pathnames";
                        descr = "Error 403.1: Relative Pathnames.";
                        body  = "Users cannot use relative pathnames to access "+
                            "folders above the main site folder.";
                        break;
                    }
                    case 2: {
                        title = "403.2 Directory Listing Denied";
                        descr = "Error 403.2: Directory Listing Denied.";
                        body  = "The system administrator has disabled the "+
                            "directory listing feature.";
                        break;
                    }
                    default: {
                        title = "403 Access Denied";
                        descr = "Error 403: Access Denied.";
                        body  = "The system administrator has not allowed you to access the file.";
                        break;
                    }
                }
                break;
            }
            case 404: {
                title = "404 File Not Found";
                descr = "Error 404: File Not Found.";
                body  = "The file you requested, "+data+", cannot be found.";
                break;
            }
/*            case 408: {
                title = "408 Request Timeout";
                descr = "Error 408: Request Timeout.";
                body  = "Your client did not send a request within the timeout interval.";
                break;
            }*/
            case 415: {
                title = "415 Unsupported Media Type";
                descr = "Error 415: Unsupported Media Type.";
                body  = "The requested entity is in a format that this server does not support.";
                break;
            }
            case 500: {
                title = "500 Internal Server Error";
                descr = "Error 500: Internal Server Error.";
                body  = "This server has suffered an internal error while processing "+
                    "a command \""+m_sCommand+"\".";
                break;
            }
            case 501: {
                title = "501 Not Implemented";
                descr = "Error 501: Method Unimplemented.";
                body  = "Your web browser sent a "+m_sCommand+" command; the only "+
                    "commands this server implements are GET and HEAD.";
                break;
            }
            case 503: {
                title = "503 Service Unavailable";
                descr = "Error 503: Service Unavailable.";
                body  = "The server on this port ("+m_scServerConf.getPort()+") has "+
                    "not been configured with any virtual hosts to serve files.  "+
                    "Please contact the system administrator.";
                break;
            }
            case 505: {
                title = "505 HTTP Version Not Supported";
                descr = "Error 505: HTTP Version Not Supported.";
                body  = "This web server does not support HTTP version "+m_sRequestVersion+".";
                break;
            }
            default: {
                title = error+" Unknown Error";
                descr = "Error "+error+": Unknown or unlisted error.";
                body  = "This HTTP request resulted in an unrecognized error being signalled.  Dump: <I>method="+m_sCommand+"; data="+data+"</I>.";
                break;
            }
        }
        /*  ERROR FORMAT
            ============

            We want to print the big error in a grayshaded table and the explanation
            in arial underneath.

            <TITLE> *** Insert Title Here *** </TITLE>
            <BODY BGCOLOR="#FFFFFF">
            <TABLE CELLBORDER=0 CELLSPACING=0 CELLPADDING=0 BORDER=0>
            <TR><TD BGCOLOR="#CCCCCC"><FONT SIZE=+2 FACE="ARIAL">Error 404: File Not Found.</FONT></TD></TR>
            </TABLE><BR><FONT FACE="ARIAL">The file you requested, /$VFS:/sigerr, cannot be found.<BR><HR ALIGN=LEFT><I><FONT SIZE=-1>
            DWong's HTTP server 4.01 at enterprise:80. Maintained by: <A HREF="mailto:MisterBean@zdnetmail.com?subject=URL:http://enterprise/$VFS:/sigerr Error: 404.0...">MisterBean@zdnetmail.com</A>.</FONT>
        */
        //this goes in the body...first init buffer
        StringBuffer sb = new StringBuffer();
        //now write stuff.
        sb.append("<HTML><HEAD><TITLE>");
        sb.append(title);
        sb.append("</TITLE></HEAD><BODY BGCOLOR=\"#FFFFFF\"><TABLE CELLBORDER=0 CELLSPACING=0 BORDER=0>");

        sb.append("<TR><TD BGCOLOR=\"#CCCCCC\"><FONT SIZE=+2 FACE=\"Arial\">");
        sb.append(descr);
        sb.append("</FONT></TD></TR></TABLE><BR><FONT FACE=\"Arial\">");

        sb.append(body);
        sb.append("<BR><HR ALIGN=LEFT><I><FONT SIZE=-1>");
        sb.append(m_hcHttpdConf.sProdName);
        sb.append(" at \"");

        if( (m_vhcHostConf != null) && (!m_vhcHostConf.getWebAdminEmail().equals("")) ) {
            sb.append(m_vhcHostConf.getHostName());
            sb.append("\" (really \"");
            sb.append(m_iaThisMachine.getHostName());
            sb.append("\" ) :");
            sb.append(m_scServerConf.getPort());

            sb.append(".  Maintained by: <A HREF=\"mailto:");
            sb.append(m_vhcHostConf.getWebAdminEmail());
            sb.append("?subject=URL:http://");
            sb.append(m_vhcHostConf.getHostName());
            sb.append(getPortAsString());
            sb.append(data);
            sb.append(" Error: ");
            sb.append(error);
            sb.append(".");
            sb.append(subcode);
            sb.append("...\">");
            sb.append(m_vhcHostConf.getWebAdminEmail());
            sb.append("</A>.");
        } else if( (m_scServerConf != null) && (!m_scServerConf.getSysAdminEmail().equals("")) ) {
            sb.append(m_iaThisMachine.getHostName());
            sb.append("\" :");
            sb.append(m_scServerConf.getPort());

            sb.append(".  Maintained by: <A HREF=\"mailto:");
            sb.append(m_scServerConf.getSysAdminEmail());
            sb.append("?subject=URL:http://");
            sb.append(m_iaThisMachine.getHostName());
            sb.append(getPortAsString());            
            sb.append(data);
            sb.append(" Error: ");
            sb.append(error);
            sb.append(".");
            sb.append(subcode);
            sb.append("...\">");
            sb.append(m_scServerConf.getSysAdminEmail());
            sb.append("</A>.");
        } else {
            sb.append(".");
        }
        sb.append("</BODY></HTML>");

        // *** WRITE DATA ***
        try {
            writeResponse();
            addResponseHeader("Content-Length", ""+sb.length());
            addResponseHeader("Content-Type", "text/html");
            writeHeaders();
            writeString(sb.toString());
        } catch (IOException e) {
            dumpException(e);
        }
    }

    public void dumpException(Throwable e) {
        if(m_bGotServer) {
            m_scServerConf.dumpException(getName(), e);
        } else {
            m_hcHttpdConf.dumpException(getName(), e);
        }
    }

    public void printDebugMessage(int level, String x) {
//        if(level <= m_scServerConf.getDebugLevel()) return;
//        if(m_bGotServer) {
        if(m_scServerConf != null) {
            m_scServerConf.println(getName(), x);
        } else {
            m_hcHttpdConf.println(getName(), x);
        }
    }

    public VHostConf getHostConf() {return m_vhcHostConf;}
    public httpdconf getHttpdConf() {return m_hcHttpdConf;}

    public void writeResponse() throws IOException {m_dosResponseStream.writeBytes(m_sResponse);}
    public void writeHeaders() throws IOException {
        m_dosResponseStream.writeBytes(m_sbHeaders.toString());
        m_dosResponseStream.writeBytes(crlf);
    }
    public void writeString(String x) throws IOException {m_dosResponseStream.writeBytes(x);}
    public void writeStream(InputStream in) throws IOException {RequestHelper.pipeData(in, m_dosResponseStream, m_vhcHostConf.getBufferSize());}
    public void writeStream(InputStream in, long length) throws IOException {RequestHelper.pipeData(in, m_dosResponseStream, length, m_vhcHostConf.getBufferSize());}

    public OutputStream getWriteStream() {return m_dosResponseStream;}
    public InputStream getReadStream() {return m_disRequestStream;}

    public InetAddress getRemoteAddr() {return m_iaClient;}
    public int getRemotePort() {return m_Socket.getPort();}
    public int getHostPort() {return m_scServerConf.getPort();}

    public String getPortAsString() {
        if(m_scServerConf == null) return ""; //if no server, no port.
        if(m_scServerConf.getPort() == 80) return ""; //http default port is 80; don't need to report that.
        else return ":"+m_scServerConf.getPort();
    }

    public void setStatusCode(int status) {
        m_sResponse = "HTTP/1.0 "+status+" ";
        switch(status) {
            case -1: m_sResponse = "HTTP/1.0 200 OK"; break;        
            case 200: m_sResponse += "OK"; break;
//            case 201: m_sResponse += "Created"; break;
//            case 202: m_sResponse += "Accepted"; break;
            case 204: m_sResponse += "No Content"; break;
//            case 205: m_sResponse += "Reset Content"; break; //http1.1
//            case 206: m_sResponse += "Partial Content"; break; //http1.1
//            case 300: m_sResponse += "Multiple Choices"; break; //requires Location: header
//            case 301: m_sResponse += "Moved Permanently"; break; //note: requires a Location: header
            case 302: m_sResponse += "Moved Temporarily"; break; //note: requires a Location: header
//            case 303: m_sResponse += "See Other"; break; //http1.1, uses Location:
            case 304: m_sResponse += "Not Modified"; break; //use with Last-Modified: header
//            case 305: m_sResponse += "Use Proxy"; break; //http1.1, use Location:
            case 400: m_sResponse += "Bad Request"; break;
//            case 401: m_sResponse += "Unauthorized"; break; //note: requires a WWW-Authenticate: header
//            case 402: m_sResponse += "Payment Required"; break; //http1.1
            case 403: m_sResponse += "Forbidden"; break;
            case 404: m_sResponse += "Not Found"; break;
//            case 405: m_sResponse += "Method Not Allowed"; break; //http1.1
//            case 406: m_sResponse += "Not Acceptable"; break; //http1.1
//            case 407: m_sResponse += "Proxy Authentication Required"; break; //http1.1
            case 408: m_sResponse += "Request Timeout"; break; //http1.1
//            case 409: m_sResponse += "Conflict"; break; //http1.1
//            case 410: m_sResponse += "Gone"; break; //http1.1
//            case 411: m_sResponse += "Length Required"; break; //http1.1, use with POST...
//            case 412: m_sResponse += "Precondition Failed"; break; //http1.1
//            case 413: m_sResponse += "Request Entity Too Large"; break; //http1.1
//            case 414: m_sResponse += "Request URI Too Long"; break; //http1.1
//            case 415: m_sResponse += "Unsupported Media Type"; break; //http1.1
            case 500: m_sResponse += "Internal Server Error"; break;
            case 501: m_sResponse += "Not Implemented"; break;
//            case 502: m_sResponse += "Bad Gateway"; break;
            case 503: m_sResponse += "Service Unavailable"; break; //http1.1, use with Retry-After:
//            case 504: m_sResponse += "Gateway Timeout"; break; //http1.1
            case 505: m_sResponse += "HTTP Version Not Supported"; break; //http1.1
            default: {
                throw new IllegalArgumentException("Status codes range from 200 to 503; "+status+" is not within that range.");
            }
        }
        printDebugMessage(3, "Response: "+m_sResponse);
        m_sResponse += "\r\n";
    }

    public void addResponseHeader(String str) {
        m_sbHeaders.append(str);
        m_sbHeaders.append(crlf);
        printDebugMessage(3, "Added header to output: "+str);
    }

    public void addResponseHeader(String name, String value) {
        m_sbHeaders.append(name);
        m_sbHeaders.append(": ");
        m_sbHeaders.append(value);
        m_sbHeaders.append(crlf);
        printDebugMessage(3, "Added header to output: "+name+": "+value);
    }

    public void dumpConfig(StringBuffer result) {
        result.append("<H2>Request Handler \""+this+"\":</H2>\r\n");
        result.append("<UL>\r\n");
        result.append("<LI>Is in the thread pool? "+m_bAddedToThreadQueue+"</LI>\r\n");
        result.append("<LI>Is handling request? "+m_bHasData+"</LI>\r\n");
        if(m_bHasData) {
            result.append("Request Data:\r\n<UL>");
            result.append("<LI>Command = "+m_sCommand+"</LI>\r\n");
            result.append("<LI>HTTP Request Version = "+m_sRequestVersion+"</LI>\r\n");
            result.append("<LI>Response = "+m_sResponse+"</LI>\r\n");
//            result.append("<LI>URL = "+m_sURL+"</LI>\r\n");
//            result.append("<LI>Local file = "+m_sLocalFile+"</LI>\r\n");
            result.append("<LI>Client = "+m_iaClient+"</LI>\r\n");
//            result.append("<LI>Content Type = "+m_sContentType+"</LI>\r\n");

              result.append("<LI>Headers:</LI>\r\n");
              result.append("<UL>\r\n");
              Enumeration e = m_prHeaders.keys();
              while(e.hasMoreElements()) {
                String key = (String) e.nextElement();
                String value = m_prHeaders.getProperty(key);
                result.append("<LI>\""+key+"\" = \""+value+"\"</LI>\r\n");
              }
              result.append("</UL>\r\n");
            result.append("</UL>\r\n");
        }
        result.append("</UL>\r\n");
    }

    public boolean isSupportedHttpVersion() {
        if(m_sRequestVersion == null) return true;
        String major, minor;
        int period, maj, min;
        period = m_sRequestVersion.indexOf('.');
        major = m_sRequestVersion.substring(0, period);
        minor = m_sRequestVersion.substring(period+1);

        maj = Integer.parseInt(major);
        min = Integer.parseInt(minor);

        if( (maj <= iMaxMajorVersion) || (maj >= iMinMajorVersion) ) {
            switch(maj) {
                case 1: {
                    if( (min >= iMinMinorVersion) || (min <= iMaxMinorVersion) ) return true;
                    break;
                }
            }
        }
        return false;
    }

    public String getRequestHeaderValue(String key) {return getRequestHeaderValue(key, null);}
    public String getRequestHeaderValue(String key, String def) {return m_prHeaders.getProperty(key, def);}
    public Enumeration getRequestHeaderNamesEnumerator() {return m_prHeaders.propertyNames();}

    public int getResponseHeaderSize() {return m_sbHeaders.length();}

    public void redirect(String sNewUrl) {
        setStatusCode(302);
        addResponseHeader("Location", sNewUrl);

        // *** WRITE DATA ***
        try {
            writeResponse();
            writeHeaders();
        } catch (SocketException ee) {
            return;
        } catch (IOException e) {
            m_hcHttpdConf.dumpException(getName(), e);
        }
    }
}
