/*
 * ugh.c -- Main loop
 * Copyright (C) 2006 Darrick Wong
 */
#include <stdio.h>
#include <signal.h>
#include <termios.h>
#include <string.h>
#include <errno.h>
#include <time.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <sys/select.h>
#include <sys/time.h>
#include <sys/types.h>
#include <stdlib.h>
#include <unistd.h>
#include <stdint.h>
#include "ugh.h"

#define BANNER		"ugh v0.48 Copyright (C) 2006 Darrick Wong.\n"

#define UGH_UPTIME	1
#define UGH_CPUS	2
#define UGH_MEMINFO	4
#define UGH_NETDEV	8
#define UGH_DISKS	16
#define UGH_IRQ		32
#define UGH_ALL		(UGH_DISKS | UGH_UPTIME | UGH_CPUS | UGH_MEMINFO | UGH_NETDEV | UGH_IRQ)

static uint32_t features = UGH_ALL;
static int in_fd;

static int enable_line_buffer(int fd);
static int disable_line_buffer(int fd);

static int detailed_report = 1;
void toggle_details(void)
{
	detailed_report = !detailed_report;
}

int is_detailed(void)
{
	return detailed_report;
}

static struct timeval real_delay;

static void set_delay(double d, struct timeval *tv)
{
	uint64_t tmp;

	tmp = d * 1000000;
	if (tmp < 10000)
		tmp = 10000;
	tv->tv_sec = tmp / 1000000;
	tv->tv_usec = tmp % 1000000;
}

/* Do the actual processing */
static void do_processing(void)
{
	printf(ZERO_SCR);
	update_time();

	process_uname();
	if (features & UGH_UPTIME) {
		process_uptime();
		process_loadavg();
	}
	if (features & UGH_CPUS)
		process_cpus();
	if (features & UGH_MEMINFO)
		process_meminfo();
	if (features & UGH_NETDEV)
		process_netdev();
	if (features & UGH_DISKS)
		process_disk();
	if (features & UGH_IRQ)
		process_irq();
	fflush(stdout);
}

/* Handle window resize by clearing screen */
void clear_screen_signal(int sig)
{
	disable_line_buffer(in_fd);
	printf(CLEAR_SCR);
	update_windata();
	do_processing();
}

/* Toggle a bit */
static inline void toggle_bit(uint32_t *mem, uint32_t val) {
	if (*mem & val)
		*mem &= ~val;
	else
		*mem |= val;
}

/* Print help */
static void print_help(int fd)
{
	int junk;

	printf(CLEAR_SCR);
	printf(ZERO_SCR);
	printf(BANNER);
	printf("Keys: ahHDucmnrq\n");
	printf("h: Help     u: Uptime      c: CPUs  m: Memory        n: Network  d: Disks\n");
	printf("r: Refresh  a: Toggle all  q: Quit  H: Human output  D: Delay    s: Simple\n");
	printf("U: Units    i: Interrupts\n");
	printf("\n");
	printf("CPU graph: UNSiIs\n");
	printf("U: User time  N: Nice time  S: Kernel time  i: I/O wait  I: IRQ handling\n");
	printf("s: softirq\n");
	printf("\n");
	printf("Memory graph: PBCS\n");
	printf("P: Programs  B: Buffers  C: Cache  S: Swap\n");
	printf("\n");
	printf("Network graph:\n");
	printf("Left graph is bytes/sec, right graph is packets/sec.\n");
	printf("\n");
	printf("Disk graph:\n");
	printf("Left graph is sectors/sec, right graph is IOs/sec.\n");
	printf("\n");
	printf("Press any key to continue.\n");
	fcntl(fd, F_SETFL, fcntl(fd, F_GETFL) & ~O_NONBLOCK);
	read(fd, &junk, 1);
	fcntl(fd, F_SETFL, fcntl(fd, F_GETFL) | O_NONBLOCK);
}

/* Print command line options */
static void print_cmd_help(const char *progname)
{
	printf("Usage: %s [-D secs] -sUudcmni\n", progname);
	printf("secs = number of seconds to wait between refreshes.  "
		"Can be floating point.\n");
	printf("-s	Simple display.\n");
	printf("-U	Use discrete units instead of bytes.\n");
	printf("-u	Disable uptime display.\n");
	printf("-d	Disable disk display.\n");
	printf("-c	Disable CPU display.\n");
	printf("-m	Disable memory display.\n");
	printf("-n	Disable network display.\n");
	printf("-i	Disable interrupt (IRQ) display.\n");
}

/* Set flags according to keystrokes. */
static void eat_keystrokes(int fd)
{
	char c;
	double d;
	
	while (read(fd, &c, 1) == 1) {
		switch (c) {
			case 'h':
			case 'H':
				print_help(fd);
				break;
			case 'q':
				exit(0);
			case 's':
				toggle_details();
				break;
			case 'U':
				toggle_disk_units();
				toggle_net_units();
				break;
			case 'u':
				toggle_bit(&features, UGH_UPTIME);
				break;
			case 'd':
				toggle_bit(&features, UGH_DISKS);
				break;
			case 'c':
				toggle_bit(&features, UGH_CPUS);
				break;
			case 'm':
				toggle_bit(&features, UGH_MEMINFO);
				break;
			case 'n':
				toggle_bit(&features, UGH_NETDEV);
				break;
			case 'a':
				toggle_bit(&features, UGH_ALL);
				break;
			case 'i':
				toggle_bit(&features, UGH_IRQ);
				break;
			/* ^L ? */
			case 0x0C:
				printf(CLEAR_SCR);
				return;
			case 'D':
				enable_line_buffer(in_fd);
				printf("New delay: ");
				FILE *fp = fopen("/dev/tty", "r");
				if (!fp) {
					printf("Cannot read?!\n");
					return;
				}
				fscanf(fp, "%lf", &d);
				fclose(fp);
				set_delay(d, &real_delay);
				disable_line_buffer(in_fd);
				break;
			case '\n':
			case 'r':
				return;
			default:
				printf("Don't know what to do with '%c' (0x%x)!\n", c, c);
				return;
		}
	}
	printf(CLEAR_SCR);
}

/* Enable line buffering */
static int enable_line_buffer(int fd)
{
	struct termios term;
	int x;

	x = tcgetattr(fd, &term);
	if (x)
		return 0;
	term.c_lflag |= ICANON | ECHO;
	x = tcsetattr(fd, TCSANOW, &term);
	if (x)
		return 0;
	x = fcntl(fd, F_SETFL, fcntl(fd, F_GETFL) & ~O_NONBLOCK);
	if (x)
		return 0;
	return 1;
}

/* Restore tty at exit */
static void restore_tty(void)
{
	enable_line_buffer(in_fd);
}

/* Disable line buffering */
static int disable_line_buffer(int fd)
{
	struct termios term;
	int x;

	x = tcgetattr(fd, &term);
	if (x)
		return 0;
	term.c_lflag &= ~ICANON;
	term.c_lflag &= ~ECHO;
	x = tcsetattr(fd, TCSANOW, &term);
	if (x)
		return 0;
	x = fcntl(fd, F_SETFL, fcntl(fd, F_GETFL) | O_NONBLOCK);
	if (x)
		return 0;
	return 1;
}

int main(int argc, char *argv[])
{
	int c;
	struct timeval init_delay, junk;
	struct timeval *delay = &init_delay;
	struct sigaction zig;
	fd_set fds;

	real_delay.tv_sec = 1;
	real_delay.tv_usec = 0;
	init_delay.tv_sec = 0;
	init_delay.tv_usec = 100000;

	printf(BANNER);

	while ( (c = getopt(argc, argv, "D:sUudcmnai")) > 0) {
		switch (c) {
			case 'D':
				set_delay(strtod(optarg, NULL), &real_delay);
				break;
			case 's':
				toggle_details();
				break;
			case 'U':
				toggle_disk_units();
				toggle_net_units();
				break;
			case 'u':
				toggle_bit(&features, UGH_UPTIME);
				break;
			case 'd':
				toggle_bit(&features, UGH_DISKS);
				break;
			case 'c':
				toggle_bit(&features, UGH_CPUS);
				break;
			case 'm':
				toggle_bit(&features, UGH_MEMINFO);
				break;
			case 'n':
				toggle_bit(&features, UGH_NETDEV);
				break;
			case 'i':
				toggle_bit(&features, UGH_IRQ);
				break;
			default:
				print_cmd_help(argv[0]);
				return 0;
		}
	}

	if (!init_window() || !init_time() || !init_uname())
		return 1;

	in_fd = open("/dev/tty", O_RDONLY | O_NONBLOCK);
	if (in_fd < 0) {
		perror("/dev/tty");
		return 1;
	}
	if (!disable_line_buffer(in_fd)) {
		perror("/dev/tty");
		return 1;
	}

	printf(CLEAR_SCR);

	memset(&zig, 0, sizeof(struct sigaction));
	zig.sa_handler = clear_screen_signal;
	sigaction(SIGWINCH, &zig, NULL);
	sigaction(SIGCONT, &zig, NULL);
	atexit(restore_tty);

	while (1) {
		do_processing();
		FD_ZERO(&fds);
		FD_SET(in_fd, &fds);
		junk = *delay;
		c = select(in_fd + 1, &fds, NULL, NULL, &junk);
		if (c > 0) {
			eat_keystrokes(in_fd);
		} else if (c < 0 && errno != EINTR) {
			perror("select");
			return 1;
		}
		delay = &real_delay;
	}

	return 0;
}
