/*
 * net.c -- Suck network transfer data out of /proc/net/dev
 * Copyright (C) 2006 Darrick Wong
 */
#include <stdio.h>
#include <errno.h>
#include <inttypes.h>
#include <stdint.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <math.h>
#include "ugh.h"

#define NETDEV_CAP 4
#define BUF_SIZE 32

struct netdev_usage_stat {
	char name[BUF_SIZE];

	uint64_t rx_bytes;
	uint64_t rx_packets;
	uint64_t tx_bytes;
	uint64_t tx_packets;

	uint64_t rx_bytes_max_rate;
	uint64_t rx_packets_max_rate;
	uint64_t tx_bytes_max_rate;
	uint64_t tx_packets_max_rate;
};

struct netdev_usage {
	struct netdev_usage_stat overall;
	struct netdev_usage_stat *netdevs;
	unsigned int netdev_count, netdev_capacity;
};

static struct netdev_usage usage_info[2];
static struct netdev_usage *old_info, *new_info;

static int use_bytes = 1;
void toggle_net_units(void)
{
	use_bytes = !use_bytes;
}

/* Grab network device usage data from /proc/net/dev */
static int get_netdev_usage(void)
{
	FILE *fp;
	int x;
	unsigned int i;
	char buf[1024];
	char *c;
	struct netdev_usage_stat *stat;
	uint64_t junk;

	/* Alloc memory */
	if (!new_info) {
		if (old_info)
			new_info = &usage_info[1];
		else
			new_info = &usage_info[0];
		new_info->netdev_capacity = NETDEV_CAP;
		new_info->netdevs = calloc(NETDEV_CAP,
			sizeof (struct netdev_usage));
		if (!new_info->netdevs) {
			perror("malloc new_info_netdevs");
			return 0;
		}
	}

	fp = fopen("/proc/net/dev", "r");
	if (!fp) {
		perror("/proc/net/dev");
		return 0;
	}

	/* Ignore the first two lines */
	if (!fgets(buf, 1024, fp) || !fgets(buf, 1024, fp)) {
		perror("/proc/net/dev");
		fclose(fp);
		return 0;
	}

	memset(&new_info->overall, 0, sizeof (struct netdev_usage_stat));
	strcpy(new_info->overall.name, "   net");

	new_info->netdev_count = 0;
	do {
		/* Grab the next line */
		if (!fgets(buf, 1024, fp))
			break;
		
		/* Allocate space */
		i = new_info->netdev_count;
		if (i >= new_info->netdev_capacity) {
			new_info->netdevs = realloc(new_info->netdevs,
				(i + 1) * sizeof(struct netdev_usage_stat));
			if (!new_info->netdevs) {
				fclose(fp);
				return 0;
			}
			memset(new_info->netdevs + new_info->netdev_capacity,
				0,
				(i + 1 - new_info->netdev_capacity) *
				sizeof (struct netdev_usage_stat));
			new_info->netdev_capacity = i + 1;
		}

		stat = &new_info->netdevs[i];

		/* Find the device name */
		c = strchr(buf, ':');
		if (!c) {
			perror("Cannot find device name!");
			fclose(fp);
			return 0;
		}
		memcpy(stat->name, buf,
			(c - buf > BUF_SIZE ? BUF_SIZE : c - buf)
		);

		x = sscanf(c + 1, "%"PRIu64" %"PRIu64" %"PRIu64" %"PRIu64" %"PRIu64" %"PRIu64" %"PRIu64" %"PRIu64" %"PRIu64" %"PRIu64" %"PRIu64" %"PRIu64" %"PRIu64" %"PRIu64" %"PRIu64" %"PRIu64,
			&stat->rx_bytes,
			&stat->rx_packets,
			&junk,
			&junk,
			&junk,
			&junk,
			&junk,
			&junk,
			&stat->tx_bytes,
			&stat->tx_packets,
			&junk,
			&junk,
			&junk,
			&junk,
			&junk,
			&junk);
		if (x != 16) {
			fclose(fp);
			return 0;
		}

		new_info->overall.rx_bytes += stat->rx_bytes;
		new_info->overall.rx_packets += stat->rx_packets;
		new_info->overall.tx_bytes += stat->tx_bytes;
		new_info->overall.tx_packets += stat->tx_packets;
		
		new_info->netdev_count++;
	} while (1);

	fclose(fp);
	return 1;
}

/* Switch the pointers */
static void switch_pointers(void)
{
	struct netdev_usage *tmp;

	tmp = new_info;
	new_info = old_info;
	old_info = tmp;
}

#define max(a, b) ((a) > (b) ? (a) : (b))

/* Calculate the new network flow data */
static void calc_time_diff(struct netdev_usage_stat *result,
	struct netdev_usage_stat *new,
	struct netdev_usage_stat *old,
	uint64_t time_delta)
{
	memcpy(result->name, new->name, BUF_SIZE);
	result->rx_bytes = (new->rx_bytes - old->rx_bytes) / ((float)time_delta / 1000);
	result->rx_packets = (new->rx_packets - old->rx_packets) / ((float)time_delta / 1000);
	result->tx_bytes = (new->tx_bytes - old->tx_bytes) / ((float)time_delta / 1000);
	result->tx_packets = (new->tx_packets - old->tx_packets) / ((float)time_delta / 1000);

	result->rx_bytes_max_rate   = max(old->rx_bytes_max_rate,
		result->rx_bytes);
	result->rx_packets_max_rate = max(old->rx_packets_max_rate,
		result->rx_packets);
	result->tx_bytes_max_rate   = max(old->tx_bytes_max_rate,
		result->tx_bytes);
	result->tx_packets_max_rate = max(old->tx_packets_max_rate,
		result->tx_packets);

	new->rx_bytes_max_rate = result->rx_bytes_max_rate;
	new->rx_packets_max_rate = result->rx_packets_max_rate;
	new->tx_bytes_max_rate = result->tx_bytes_max_rate;
	new->tx_packets_max_rate = result->tx_packets_max_rate;
}

static void print_device(struct netdev_usage_stat *stat)
{
	printf("%s: %"PRIu64" %"PRIu64" %"PRIu64" %"PRIu64" %"PRIu64" %"PRIu64" %"PRIu64" %"PRIu64,
		stat->name,
		stat->rx_bytes,
		stat->rx_packets,
		stat->rx_bytes_max_rate,
		stat->rx_packets_max_rate,

		stat->tx_bytes,
		stat->tx_packets,
		stat->tx_bytes_max_rate,
		stat->tx_packets_max_rate
	      );
}

#define Y_FUNC(a, b, c)	( (a) ? my_round( ((double)(a) / (b)) * (c) ) : 0)

/* Print network device stats */
static void print_time_scale(struct netdev_usage_stat *stat,
	unsigned int chars)
{
	unsigned int x, y, z;
	unsigned int gchars;

	if (use_bytes)
		z = printf("%s: R %6.0f/%6.0f K/s ", stat->name,
		(((float)stat->rx_bytes / 1024)), (((float)stat->rx_bytes_max_rate / 1024)));
	else
		z = printf("%s: R %6"PRIu64"/%6"PRIu64" pkt ", stat->name,
		stat->rx_packets, stat->rx_packets_max_rate);

	gchars = (chars - z - 2) / 2;
	z = 0;
	y = Y_FUNC(stat->rx_bytes, stat->rx_bytes_max_rate, gchars);

	for (x = 0; x < y && z < gchars; x++, z++)
		printf("R");
	
	if (z < gchars)
		for (x = gchars - z; x > 0; x--)
			printf("-");
	printf(" ");

	z = 0;
	y = Y_FUNC(stat->rx_packets, stat->rx_packets_max_rate, gchars);
	for (x = 0; x < y && z < gchars; x++, z++)
		printf("r");
	
	if (z < gchars)
		for (x = gchars - z; x > 0; x--)
			printf("-");
	printf("\n");

	if (use_bytes)
		z = printf("        T %6.0f/%6.0f K/s ",
		(((float)stat->tx_bytes / 1024)), (((float)stat->tx_bytes_max_rate / 1024)));
	else
		z = printf("        T %6"PRIu64"/%6"PRIu64" pkt ",
		stat->tx_packets, stat->tx_packets_max_rate);

	gchars = (chars - z - 2) / 2;
	z = 0;
	y = Y_FUNC(stat->tx_bytes, stat->tx_bytes_max_rate, gchars);
	for (x = 0; x < y && z < gchars; x++, z++)
		printf("T");
	
	if (z < gchars)
		for (x = gchars - z; x > 0; x--)
			printf("-");
	printf(" ");

	z = 0;
	y = Y_FUNC(stat->tx_packets, stat->tx_packets_max_rate, gchars);
	for (x = 0; x < y && z < gchars; x++, z++)
		printf("t");
	
	if (z < gchars)
		for (x = gchars - z; x > 0; x--)
			printf("-");
	printf("\n");

#if 0
	print_device(stat);
	printf("\n");
#endif
}

#undef Y_FUNC

/* Find a network device with a particular name */
static struct netdev_usage_stat *find_device(const char *name,
	struct netdev_usage *nu)
{
	int i;

	for (i = 0; i < nu->netdev_count; i++) {
		if (!strcmp(nu->netdevs[i].name, name))
			return &nu->netdevs[i];
	}

	return NULL;
}

/* Display the network device data */
static void display_netdev_usage(void)
{
	struct netdev_usage_stat tmp;
	struct netdev_usage_stat *old;
	uint64_t time_delta;
	unsigned int i, to_display;
	
	if (!new_info || !old_info)
		return;

	time_delta = elapsed_time();

	calc_time_diff(&tmp, &new_info->overall, &old_info->overall, time_delta);
	print_time_scale(&tmp, get_window_width());

	to_display = 0;
	for (i = 0; i <= new_info->netdev_count; i++) {
		old = find_device(new_info->netdevs[i].name, old_info);
		if (!old)
			continue;
		calc_time_diff(&tmp, &new_info->netdevs[i], old, time_delta);
		if (tmp.rx_bytes_max_rate ||
			tmp.rx_packets_max_rate ||
			tmp.tx_bytes_max_rate ||
			tmp.tx_packets_max_rate)
			to_display++;
	}

	if (to_display < 2 || !is_detailed())
		return;

	for (i = 0; i <= new_info->netdev_count; i++) {
		old = find_device(new_info->netdevs[i].name, old_info);
		if (!old)
			continue;
		calc_time_diff(&tmp, &new_info->netdevs[i], old, time_delta);
		if (tmp.rx_bytes_max_rate ||
			tmp.rx_packets_max_rate ||
			tmp.tx_bytes_max_rate ||
			tmp.tx_packets_max_rate)
			print_time_scale(&tmp, get_window_width());
	}
}

/* Process network device data */
void process_netdev(void)
{
	if (!get_netdev_usage())
		return;

	display_netdev_usage();
	
	switch_pointers();
}
