/*
 * irq.c -- Report IRQ data.
 * Copyright (C) 2006 Darrick Wong
 */
#include <stdio.h>
#include <errno.h>
#include <inttypes.h>
#include <stdint.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <math.h>
#include "ugh.h"

#define IRQ_CAP 20
#define BUF_SIZE 32

struct irq_usage_stat {
	char name[BUF_SIZE];

	uint64_t count;
	uint64_t max_rate;
};

struct irq_usage {
	struct irq_usage_stat overall;
	struct irq_usage_stat *irqs;
	unsigned int irq_count, irq_capacity;
};

static struct irq_usage usage_info[2];
static clockid_t clock_type = CLOCK_MONOTONIC;
static struct irq_usage *old_info, *new_info;

/* Grab IRQ usage information from /proc/interrupts */
static int get_irq_usage(void)
{
	FILE *fp;
	unsigned int i;
	char buf[1024];
	char *c, *d;
	struct irq_usage_stat *stat;
	uint64_t junk;

	/* Alloc memory */
	if (!new_info) {
		if (old_info)
			new_info = &usage_info[1];
		else
			new_info = &usage_info[0];
		new_info->irq_capacity = IRQ_CAP;
		new_info->irqs = calloc(IRQ_CAP,
			sizeof (struct irq_usage));
		if (!new_info->irqs) {
			perror("malloc new_info_irqs");
			return 0;
		}
	}

	fp = fopen("/proc/interrupts", "r");
	if (!fp) {
		perror("/proc/interrupts");
		return 0;
	}

	/* Ignore the first line */
	if (!fgets(buf, 1024, fp)) {
		perror("/proc/interrupts");
		fclose(fp);
		return 0;
	}

	memset(&new_info->overall, 0, sizeof (struct irq_usage_stat));
	strcpy(new_info->overall.name, "irq");

	new_info->irq_count = 0;
	do {
		/* Grab the next line */
		if (!fgets(buf, 1024, fp))
			break;
		
		/* Allocate space */
		i = new_info->irq_count;
		if (i >= new_info->irq_capacity) {
			new_info->irqs = realloc(new_info->irqs,
				(i + 1) * sizeof(struct irq_usage_stat));
			if (!new_info->irqs) {
				fclose(fp);
				return 0;
			}
			memset(new_info->irqs + new_info->irq_capacity,
				0,
				(i + 1 - new_info->irq_capacity) *
				sizeof (struct irq_usage_stat));
			new_info->irq_capacity = i + 1;
		}

		stat = &new_info->irqs[i];

		/* Find the interrupt name */
		c = strchr(buf, ':');
		if (!c) {
			perror("Cannot find irq name!");
			fclose(fp);
			return 0;
		}
		memcpy(stat->name, buf,
			(c - buf > BUF_SIZE ? BUF_SIZE : c - buf)
		);
		stat->count = 0;
		c++;

		do {
			junk = strtoll(c, &d, 0);
			if (c == d)
				break;
			c = d;
			stat->count += junk;
		} while (1);

		stat->count += junk;

		new_info->overall.count += stat->count;
		
		new_info->irq_count++;
	} while (1);

	fclose(fp);
	return 1;
}

/* Switch the pointers */
static void switch_pointers(void)
{
	struct irq_usage *tmp;

	tmp = new_info;
	new_info = old_info;
	old_info = tmp;
}

#define max(a, b) ((a) > (b) ? (a) : (b))

/* Calculate the new interrupt data */
static void calc_time_diff(struct irq_usage_stat *result,
	struct irq_usage_stat *new,
	struct irq_usage_stat *old,
	uint64_t time_delta)
{
	memcpy(result->name, new->name, BUF_SIZE);
	result->count = (new->count - old->count) / ((float)time_delta / 1000);

	result->max_rate   = max(old->max_rate, result->count);

	new->max_rate = result->max_rate;
}

static void print_device(struct irq_usage_stat *stat)
{
	printf("%s: %"PRIu64" %"PRIu64,
		stat->name,
		stat->count,
		stat->max_rate
	      );
}

#define Y_FUNC(a, b, c)	( (a) ? my_round( ((double)(a) / (b)) * (c) ) : 0)

/* Print interrupt stats */
static void print_time_scale(struct irq_usage_stat *stat,
	unsigned int chars)
{
	unsigned int x, y, z;

	chars--;
	z = printf("%6s: %6"PRIu64"/%6"PRIu64" irq/s ", stat->name,
		stat->count, stat->max_rate);

	y = Y_FUNC(stat->count, stat->max_rate, chars - z);

	for (x = 0; x < y && z < chars; x++, z++)
		printf("*");
	
	if (z < chars)
		for (x = chars - z; x > 0; x--)
			printf("-");
	printf("\n");

#if 0
	print_device(stat);
	printf("\n");
#endif
}

#undef Y_FUNC

/* Find a interrupt with a particular name */
static struct irq_usage_stat *find_device(const char *name,
	struct irq_usage *nu)
{
	int i;

	for (i = 0; i < nu->irq_count; i++) {
		if (!strcmp(nu->irqs[i].name, name))
			return &nu->irqs[i];
	}

	return NULL;
}

/* Display the interrupt data */
static void display_irq_usage(void)
{
	struct irq_usage_stat tmp;
	struct irq_usage_stat *old;
	uint64_t time_delta;
	unsigned int i, to_display;
	
	if (!new_info || !old_info)
		return;

	time_delta = elapsed_time();

	calc_time_diff(&tmp, &new_info->overall, &old_info->overall, time_delta);
	print_time_scale(&tmp, get_window_width());

	to_display = 0;
	for (i = 0; i <= new_info->irq_count; i++) {
		old = find_device(new_info->irqs[i].name, old_info);
		if (!old)
			continue;
		calc_time_diff(&tmp, &new_info->irqs[i], old, time_delta);
		if (tmp.count || tmp.max_rate)
			to_display++;
	}

	if (to_display < 2 || !is_detailed())
		return;

	for (i = 0; i <= new_info->irq_count; i++) {
		old = find_device(new_info->irqs[i].name, old_info);
		if (!old)
			continue;
		calc_time_diff(&tmp, &new_info->irqs[i], old, time_delta);
		if (tmp.count || tmp.max_rate)
			print_time_scale(&tmp, get_window_width());
	}
}

/* Process interrupt data */
void process_irq(void)
{
	if (!get_irq_usage())
		return;

	display_irq_usage();
	
	switch_pointers();
}
