/*
 * disk.c -- Grab disk data out of /proc/diskstats
 * Copyright (C) 2006 Darrick Wong
 */
#include <stdio.h>
#include <errno.h>
#include <inttypes.h>
#include <stdint.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <math.h>
#include "ugh.h"

#define DISK_CAP 4
#define BUF_SIZE 32

struct disk_usage_stat {
	char name[BUF_SIZE];

	uint64_t rd_ios;
	uint64_t rd_sectors;
	uint64_t rd_ticks;
	uint64_t wr_ios;
	uint64_t wr_sectors;
	uint64_t wr_ticks;

	uint64_t rd_ios_max_rate;
	uint64_t rd_sectors_max_rate;
	uint64_t wr_ios_max_rate;
	uint64_t wr_sectors_max_rate;
};

struct disk_usage {
	struct disk_usage_stat overall;
	struct disk_usage_stat *disks;
	unsigned int disk_count, disk_capacity;
};

static struct disk_usage usage_info[2];
static clockid_t clock_type = CLOCK_MONOTONIC;
static struct disk_usage *old_info, *new_info;

static int use_bytes = 1;
void toggle_disk_units(void)
{
	use_bytes = !use_bytes;
}

/* Grab disk device usage data from /proc/diskstats */
static int get_disk_usage(void)
{
	FILE *fp;
	int x;
	unsigned int i;
	char buf[1024];
	struct disk_usage_stat temp;
	struct disk_usage_stat *stat;
	uint64_t junk;

	/* Alloc memory */
	if (!new_info) {
		if (old_info)
			new_info = &usage_info[1];
		else
			new_info = &usage_info[0];
		new_info->disk_capacity = DISK_CAP;
		new_info->disks = calloc(DISK_CAP,
			sizeof (struct disk_usage));
		if (!new_info->disks) {
			perror("malloc new_info_disks");
			return 0;
		}
	}

	fp = fopen("/proc/diskstats", "r");
	if (!fp) {
		perror("/proc/diskstats");
		return 0;
	}

	memset(&new_info->overall, 0, sizeof (struct disk_usage_stat));
	strcpy(new_info->overall.name, " disks");

	new_info->disk_count = 0;
	do {
		/* Grab the next line */
		if (!fgets(buf, 1024, fp))
			break;

		memset(&temp, 0, sizeof(struct disk_usage_stat));
		
		/* Count/read fields to determine blockdev type */
		x = sscanf(buf, "%"PRIu64" %"PRIu64" %s %"PRIu64" %"PRIu64" %"PRIu64" %"PRIu64" %"PRIu64" %"PRIu64" %"PRIu64" %"PRIu64" %"PRIu64" %"PRIu64" %"PRIu64,
			&junk,
			&junk,
			temp.name,
			&temp.rd_ios,
			&junk,
			&temp.rd_sectors,
			&temp.rd_ticks,
			&temp.wr_ios,
			&junk,
			&temp.wr_sectors,
			&temp.wr_ticks,
			&junk,
			&junk,
			&junk);
		if (x == 7)
			continue;
		
		/* Allocate space */
		i = new_info->disk_count;
		if (i >= new_info->disk_capacity) {
			new_info->disks = realloc(new_info->disks,
				(i + 1) * sizeof(struct disk_usage_stat));
			if (!new_info->disks) {
				fclose(fp);
				return 0;
			}
			memset(new_info->disks + new_info->disk_capacity,
				0,
				(i + 1 - new_info->disk_capacity) *
				sizeof (struct disk_usage_stat));
			new_info->disk_capacity = i + 1;
		}

		/* Copy all the stat data */
		memcpy(&new_info->disks[i], &temp, sizeof(struct disk_usage_stat));
		stat = &temp;

		new_info->overall.rd_ios += stat->rd_ios;
		new_info->overall.rd_sectors += stat->rd_sectors;
		new_info->overall.rd_ticks += stat->rd_ticks;
		new_info->overall.wr_ios += stat->wr_ios;
		new_info->overall.wr_sectors += stat->wr_sectors;
		new_info->overall.wr_ticks += stat->wr_ticks;
		
		new_info->disk_count++;
	} while (1);

	fclose(fp);
	return 1;
}

/* Switch the pointers */
static void switch_pointers(void)
{
	struct disk_usage *tmp;

	tmp = new_info;
	new_info = old_info;
	old_info = tmp;
}

#define max(a, b) ((a) > (b) ? (a) : (b))

/* Calculate the new disk flow data */
static void calc_time_diff(struct disk_usage_stat *result,
	struct disk_usage_stat *new,
	struct disk_usage_stat *old,
	uint64_t time_delta)
{
	memcpy(result->name, new->name, BUF_SIZE);
	result->rd_ios = (new->rd_ios - old->rd_ios) / ((float)time_delta / 1000);
	result->rd_sectors = (new->rd_sectors - old->rd_sectors) / ((float)time_delta / 1000);
	result->rd_ticks = new->rd_ticks - old->rd_ticks;
	result->wr_ios = (new->wr_ios - old->wr_ios) / ((float)time_delta / 1000);
	result->wr_sectors = (new->wr_sectors - old->wr_sectors) / ((float)time_delta / 1000);
	result->wr_ticks = new->wr_ticks - old->wr_ticks;

	result->rd_ios_max_rate   = max(old->rd_ios_max_rate,
		result->rd_ios);
	result->rd_sectors_max_rate = max(old->rd_sectors_max_rate,
		result->rd_sectors);
	result->wr_ios_max_rate   = max(old->wr_ios_max_rate,
		result->wr_ios);
	result->wr_sectors_max_rate = max(old->wr_sectors_max_rate,
		result->wr_sectors);

	new->rd_ios_max_rate = result->rd_ios_max_rate;
	new->rd_sectors_max_rate = result->rd_sectors_max_rate;
	new->wr_ios_max_rate = result->wr_ios_max_rate;
	new->wr_sectors_max_rate = result->wr_sectors_max_rate;
}

static void print_device(struct disk_usage_stat *stat)
{
	printf("%s: %"PRIu64" %"PRIu64" %"PRIu64" %"PRIu64" %"PRIu64" %"PRIu64" %"PRIu64" %"PRIu64,
		stat->name,
		stat->rd_ios,
		stat->rd_sectors,
		stat->rd_ios_max_rate,
		stat->rd_sectors_max_rate,

		stat->wr_ios,
		stat->wr_sectors,
		stat->wr_ios_max_rate,
		stat->wr_sectors_max_rate
	      );
}

#define Y_FUNC(a, b, c)	( (a) ? my_round( ((double)(a) / (b)) * (c) ) : 0)

/* Print disk usage graph */
static void print_time_scale(struct disk_usage_stat *stat,
	unsigned int chars)
{
	unsigned int x, y, z;
	unsigned int gchars;

	if (use_bytes)
		z = printf("%6s: R %6.0f/%6.0f K/s ", stat->name,
		(((float)stat->rd_sectors / 2)), (((float)stat->rd_sectors_max_rate / 2)));
	else
		z = printf("%6s: R %6"PRIu64"/%6"PRIu64" ios ", stat->name,
		stat->rd_ios, stat->rd_ios_max_rate);

	gchars = (chars - z - 2) / 2;
	z = 0;
	y = Y_FUNC(stat->rd_sectors, stat->rd_sectors_max_rate, gchars);
	for (x = 0; x < y && z < gchars; x++, z++)
		printf("R");
	
	if (z < gchars)
		for (x = gchars - z; x > 0; x--)
			printf("-");
	printf(" ");

	z = 0;
	y = Y_FUNC(stat->rd_ios, stat->rd_ios_max_rate, gchars);

	for (x = 0; x < y && z < gchars; x++, z++)
		printf("r");
	
	if (z < gchars)
		for (x = gchars - z; x > 0; x--)
			printf("-");
	printf("\n");

	if (use_bytes)
		z = printf("        W %6.0f/%6.0f K/s ",
		(((float)stat->wr_sectors / 2)), (((float)stat->wr_sectors_max_rate / 2)));
	else
		z = printf("        W %6"PRIu64"/%6"PRIu64" ios ",
		stat->wr_ios, stat->wr_ios_max_rate);
	gchars = (chars - z - 2) / 2;
	z = 0;
	y = Y_FUNC(stat->wr_sectors, stat->wr_sectors_max_rate, gchars);
	for (x = 0; x < y && z < gchars; x++, z++)
		printf("W");
	
	if (z < gchars)
		for (x = gchars - z; x > 0; x--)
			printf("-");
	printf(" ");

	z = 0;
	y = Y_FUNC(stat->wr_ios, stat->wr_ios_max_rate, gchars);
	for (x = 0; x < y && z < gchars; x++, z++)
		printf("w");
	
	if (z < gchars)
		for (x = gchars - z; x > 0; x--)
			printf("-");
	printf("\n");

#if 0
	print_device(stat);
	printf("\n");
#endif
}

#undef Y_FUNC

/* Find a disk device with a particular name */
static struct disk_usage_stat *find_device(const char *name,
	struct disk_usage *nu)
{
	int i;

	for (i = 0; i < nu->disk_count; i++) {
		if (!strcmp(nu->disks[i].name, name))
			return &nu->disks[i];
	}

	return NULL;
}

/* Display the disk device data */
static void display_disk_usage(void)
{
	struct disk_usage_stat tmp;
	struct disk_usage_stat *old;
	uint64_t time_delta;
	unsigned int i, to_display;
	
	if (!new_info || !old_info)
		return;

	time_delta = elapsed_time();

	calc_time_diff(&tmp, &new_info->overall, &old_info->overall, time_delta);
	print_time_scale(&tmp, get_window_width());

	to_display = 0;
	for (i = 0; i <= new_info->disk_count; i++) {
		old = find_device(new_info->disks[i].name, old_info);
		if (!old)
			continue;
		calc_time_diff(&tmp, &new_info->disks[i], old, time_delta);
		if (tmp.rd_ios_max_rate ||
			tmp.rd_sectors_max_rate ||
			tmp.wr_ios_max_rate ||
			tmp.wr_sectors_max_rate)
			to_display++;
	}

	if (to_display < 2 || !is_detailed())
		return;

	for (i = 0; i <= new_info->disk_count; i++) {
		old = find_device(new_info->disks[i].name, old_info);
		if (!old)
			continue;
		calc_time_diff(&tmp, &new_info->disks[i], old, time_delta);
		if (tmp.rd_ios_max_rate ||
			tmp.rd_sectors_max_rate ||
			tmp.wr_ios_max_rate ||
			tmp.wr_sectors_max_rate)
			print_time_scale(&tmp, get_window_width());
	}
}

/* Process disk device data */
void process_disk(void)
{
	if (!get_disk_usage())
		return;

	display_disk_usage();
	
	switch_pointers();
}
