/*
 * cpu.c -- Suck CPU time data out of /proc/stat
 * Copyright (C) 2006 Darrick Wong
 */
#include <stdio.h>
#include <inttypes.h>
#include <errno.h>
#include <stdint.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include "ugh.h"

#define CPU_CAP 32

struct cpu_usage_stat {
	uint64_t user;
	uint64_t nice;
	uint64_t system;
	uint64_t softirq;
	uint64_t irq;
	uint64_t idle;
	uint64_t iowait;
	uint64_t steal;

	uint64_t cur_freq;
	uint64_t max_freq;
};

struct cpu_usage {
	struct cpu_usage_stat overall;
	struct cpu_usage_stat *cpus;
	unsigned int cpu_count, cpu_capacity, highest_cpu;
};

static struct cpu_usage usage_info[2];

static struct cpu_usage *old_info, *new_info;

/* Grab CPU usage data from /proc/stat */
static int get_cpu_usage(void)
{
	FILE *fp, *fp2;
	int x;
	unsigned int i;
	char buf[256];

	/* Alloc memory */
	if (!new_info) {
		if (old_info)
			new_info = &usage_info[1];
		else
			new_info = &usage_info[0];
		new_info->cpu_capacity = CPU_CAP;
		new_info->cpus = calloc(CPU_CAP, sizeof (struct cpu_usage));
		if (!new_info->cpus) {
			perror("malloc new_info_cpus");
			return 0;
		}
	}

	fp = fopen("/proc/stat", "r");
	if (!fp) {
		perror("/proc/stat");
		return 0;
	}

	/* Read overall stuff */
	x = fscanf(fp, "cpu  %"PRIu64" %"PRIu64" %"PRIu64" %"PRIu64" %"PRIu64" %"PRIu64" %"PRIu64" %"PRIu64"\n",
		&new_info->overall.user,
		&new_info->overall.nice,
		&new_info->overall.system,
		&new_info->overall.idle,
		&new_info->overall.iowait,
		&new_info->overall.irq,
		&new_info->overall.softirq,
		&new_info->overall.steal);

	new_info->cpu_count = 0;
	new_info->highest_cpu = 0;
	do {
		x = fscanf(fp, "cpu%d", &i);
		if (!x)
			break;
		if (i >= new_info->cpu_capacity) {
			new_info->cpus = realloc(new_info->cpus,
				(i + 1) * sizeof(struct cpu_usage_stat));
			if (!new_info->cpus) {
				fclose(fp);
				return 0;
			}
			memset(new_info->cpus + new_info->cpu_capacity, 0,
				(i + 1 - new_info->cpu_capacity) *
				sizeof (struct cpu_usage_stat));
			new_info->cpu_capacity = i + 1;
		}

		/* Load CPU time data */
		x = fscanf(fp, " %"PRIu64" %"PRIu64" %"PRIu64" %"PRIu64" %"PRIu64" %"PRIu64" %"PRIu64" %"PRIu64"\n",
			&new_info->cpus[i].user,
			&new_info->cpus[i].nice,
			&new_info->cpus[i].system,
			&new_info->cpus[i].idle,
			&new_info->cpus[i].iowait,
			&new_info->cpus[i].irq,
			&new_info->cpus[i].softirq,
			&new_info->cpus[i].steal);

		/* Now go after cpufreq data */
		snprintf(buf, 256, "/sys/devices/system/cpu/cpu%d/cpufreq/scaling_available_frequencies", i);
		fp2 = fopen(buf, "r");
		if (!fp2) {
			new_info->cpus[i].max_freq = 0;
			goto no_cpufreq;
		}

		fscanf(fp2, "%"PRIu64, &new_info->cpus[i].max_freq);
		fclose(fp2);

		snprintf(buf, 256, "/sys/devices/system/cpu/cpu%d/cpufreq/scaling_cur_freq", i);
		fp2 = fopen(buf, "r");
		if (!fp2) {
			new_info->cpus[i].max_freq = 0;
			goto no_cpufreq;
		}

		fscanf(fp2, "%"PRIu64, &new_info->cpus[i].cur_freq);
		fclose(fp2);

no_cpufreq:
		new_info->cpu_count++;
		if (i > new_info->highest_cpu)
		       new_info->highest_cpu = i;
	} while (1);

	fclose(fp);
	return 1;
}

/* Switch the pointers */
static void switch_pointers(void)
{
	struct cpu_usage *tmp;

	tmp = new_info;
	new_info = old_info;
	old_info = tmp;
}

/* Calculate time differentials */
static void calc_time_diff(struct cpu_usage_stat *result,
	struct cpu_usage_stat *new,
	struct cpu_usage_stat *old)
{
	result->user = (new->user - old->user);
	result->nice = (new->nice - old->nice);
	result->system = (new->system - old->system);
	result->idle = (new->idle - old->idle);
	result->iowait = (new->iowait - old->iowait);
	result->irq = (new->irq - old->irq);
	result->softirq = (new->softirq - old->softirq);
	result->steal = (new->steal - old->steal);
}

/* Print CPU usage data */
static void print_time_scale(int idx, struct cpu_usage_stat *stat, unsigned int chars)
{
	uint64_t total;
	unsigned int x, y, z, gchars;

	chars--;
	total = stat->user + stat->nice + stat->system + stat->idle +
		stat->iowait + stat->irq + stat->softirq + stat->steal;
	if (!total)
		return;

	if (idx >= 0)
		z = printf("%2d: ", idx);
	else
		z = printf(" S: ");
	gchars = chars - z;
	y = my_round( ((double)stat->user / total) * gchars );
	for (x = 0; x < y && z < chars; x++, z++)
		printf("U");

	y = my_round( ((double)stat->nice / total) * gchars );
	for (x = 0; x < y && z < chars; x++, z++)
		printf("N");

	y = my_round( ((double)stat->system / total) * gchars );
	for (x = 0; x < y && z < chars; x++, z++)
		printf("S");

	y = my_round( ((double)stat->iowait / total) * gchars );
	for (x = 0; x < y && z < chars; x++, z++)
		printf("i");

	y = my_round( ((double)stat->irq / total) * gchars );
	for (x = 0; x < y && z < chars; x++, z++)
		printf("I");

	y = my_round( ((double)stat->softirq / total) * gchars );
	for (x = 0; x < y && z < chars; x++, z++)
		printf("s");
	
	if (z < chars)
		for (x = chars - z; x > 0; x--)
			printf("-");
}

static void print_cpufreq_data(struct cpu_usage_stat *stat, unsigned int chars)
{
	unsigned int x, y, z, gchars;

	if (!stat->max_freq)
		return;

	z = printf("    %4"PRIu64"/%4"PRIu64" MHz: ", stat->cur_freq / 1000, stat->max_freq / 1000);
	chars--;
	gchars = chars - z;
	y = my_round( ((double)stat->cur_freq / stat->max_freq) * gchars );
	for (x = 0; x < y && z < chars; x++, z++)
		printf("F");
	
	if (z < chars)
		for (x = chars - z; x > 0; x--)
			printf("-");

	printf("\n");
}

static void print_time(struct cpu_usage_stat *stat)
{
	printf("%"PRIu64" %"PRIu64" %"PRIu64" %"PRIu64" %"PRIu64" %"PRIu64" %"PRIu64" %"PRIu64" %"PRIu64"/%"PRIu64"\n",
		stat->user,
		stat->nice,
		stat->system,
		stat->idle,
		stat->iowait,
		stat->irq,
		stat->softirq,
		stat->steal,
		stat->cur_freq,
		stat->max_freq
	      );
}

/* Display the CPU data */
static void display_cpu_usage(void)
{
	struct cpu_usage_stat tmp;
	unsigned int i;
	
	if (!new_info || !old_info)
		return;

	calc_time_diff(&tmp, &new_info->overall, &old_info->overall);
	print_time_scale(-1, &tmp, get_window_width());
	printf("\n");

	if (new_info->highest_cpu == 0) {
		print_cpufreq_data(&new_info->cpus[0], get_window_width());
		return;
	}

	if (!is_detailed()) {
		return;
	}

	for (i = 0; i <= new_info->highest_cpu; i++) {
		calc_time_diff(&tmp, &new_info->cpus[i], &old_info->cpus[i]);
		print_time_scale(i, &tmp, get_window_width());
		printf("\n");
		print_cpufreq_data(&new_info->cpus[i], get_window_width());
	}
}

/* Process CPU data */
void process_cpus(void)
{
	if (!get_cpu_usage())
		return;

	display_cpu_usage();
	
	switch_pointers();
}
