package grid.client;
//----------------------------------------------------------
// Compute square root of large numbers using Heron's method
//----------------------------------------------------------

import grid.util.Defaults;
import grid.manager.Job;
import grid.manager.ClientManager;
import grid.worker.TaskWorker;
import grid.Workable;
import grid.manager.JobFinishedCallback;
import java.math.BigInteger;
import java.math.BigDecimal;
import java.util.Map;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Set;
import java.util.List;
import java.util.LinkedList;
import java.util.Map.Entry;
import java.rmi.Naming;
import java.io.Serializable;
import java.rmi.RemoteException;
import java.rmi.server.UnicastRemoteObject;

class BigSquareRoot /*implements Workable*/ {

	private static BigDecimal ZERO = new BigDecimal ("0");
	private static BigDecimal ONE = new BigDecimal ("1");
	private static BigDecimal TWO = new BigDecimal ("2");
	public static final int DEFAULT_MAX_ITERATIONS = 50;
	public static final int DEFAULT_SCALE = 10;

	//--------------------------
	// Get initial approximation
	//--------------------------

	private static BigDecimal getInitialApproximation (BigDecimal n) {
		BigInteger integerPart = n.toBigInteger ();
		int length = integerPart.toString ().length ();
		if ((length % 2) == 0) {
			length--;
		}
		length /= 2;
		BigDecimal guess = ONE.movePointRight (length);
		return guess;
	}

	//----------------
	// Get square root
	//----------------

	public static BigDecimal get (BigInteger n) {
		return get (new BigDecimal (n));
	}

	public static BigDecimal get (BigDecimal n) {
		BigDecimal error;
		int iterations;
		int scale = DEFAULT_SCALE;
		int maxIterations = DEFAULT_MAX_ITERATIONS;

		// Make sure n is a positive number

		if (n.compareTo (ZERO) <= 0) {
			throw new IllegalArgumentException ();
		}

		BigDecimal initialGuess = getInitialApproximation (n);
		BigDecimal lastGuess = ZERO;
		BigDecimal guess = new BigDecimal (initialGuess.toString ());

		// Iterate

		iterations = 0;
		boolean more = true;
		while (more) {
			lastGuess = guess;
			guess = n.divide(guess, scale, BigDecimal.ROUND_HALF_UP);
			guess = guess.add(lastGuess);
			guess = guess.divide (TWO, scale, BigDecimal.ROUND_HALF_UP);
			error = n.subtract (guess.multiply (guess));
			if (++iterations >= maxIterations) {
				more = false;
			}
			else if (lastGuess.equals (guess)) {
				more = error.abs ().compareTo (ONE) >= 0;
			}
		}
		return guess;

	}
/*
	private BigInteger num;
	public BigSquareRoot(BigInteger num) {
		this.num = num;
	}

	public Object execute(TaskWorker control) {
		return get(num);
	}
*/
}

/**
 * Find prime numbers.  First we try to find the sqrt -- this is the upper bound
 * on the # of integers that we have to divide into the number we're testing.
 * Then it's just a matter of splitting up the division into blocks for all servers
 * to eat.
 */
public class IsPrime extends UnicastRemoteObject implements JobFinishedCallback, Serializable {
	public static void main(String args[]) throws Exception {
		if(args.length < 3) {
			System.out.println("Usage: grid.client.IsPrime <manager> <num> <chunksize>");
			return;
		}

		String host = args[0];
		BigInteger test = new BigInteger(args[1]);
		BigInteger chunksize = new BigInteger(args[2]);

		System.setSecurityManager( new java.rmi.RMISecurityManager() );
		IsPrime tc = new IsPrime(host, test, chunksize);
		tc.engage();
	}

	private String host;
	private BigInteger test;
	private Map jz;

	public IsPrime(String host, BigInteger test, BigInteger chunksize) throws RemoteException {
		super();
		this.host = host;
		this.test = test;
		this.N_BY = chunksize;
		jz = new HashMap();
	}

	private BigInteger N_BY;
	private static final BigInteger N1 = new BigInteger("1");
	private static final BigInteger N0 = new BigInteger("0");
	private boolean haveSqrt;
	private BigInteger sqrt;

	public void engage() {

		String name = "//"+host+"/"+Defaults.managerService;
		ClientManager cm = null;
		try {
			cm = (ClientManager) Naming.lookup(name);
		} catch (Exception e) {
			System.err.println("Unable to find manager at "+host+"!	Aborting.");
			System.err.println(e.getMessage());
			return;
		}

System.out.println("About to ask for sqrt");
		//first figure out sqrt(test).
		sqrt = BigSquareRoot.get(test).toBigInteger();
		/*
		haveSqrt = false;
		Job jBsr = new Job(bsr, "sqrt", new JobFinishedCallback() implements Serializable {
			public void jobFinished(String jobID, Object result) {
				sqrt = ((BigDecimal)result).toBigInteger();
				haveSqrt = true;
			}
		});
		try {
			cm.scheduleJob(jBsr);
		} catch (Exception e) {
			e.printStackTrace();
			return;
		}

		while(haveSqrt == false) {
			try {Thread.sleep(500);}catch(Exception e){}
		}
		*/

		System.out.println("sqrt("+test+") == "+sqrt);

		hasDivisor = false;

		//Cut up the division job into blocks of 150...
		biggest_seen = N0;
		BigInteger jobCounts[] = sqrt.divideAndRemainder(N_BY);
		if(jobCounts[1].compareTo(N0) > 0) {
			//if the remainder is bigger than zero...
			//add another run.
			jobCounts[0] = jobCounts[0].add(N1);
		}

		long start = System.currentTimeMillis();
		last_update = start;
		System.out.println("Started job at "+start);
		System.out.println("Submitting "+jobCounts[0]+" jobs of size "+N_BY);
		List newJobs = new LinkedList();
		try {
			for(
				BigInteger index = N0;
				index.compareTo(jobCounts[0]) < 0;
				index = index.add(N1)
			) {
				//otherwise, keep shovelling in jobs.
				BigInteger range = N_BY.min(
					sqrt.subtract(
						index.multiply(N_BY)
					)
				);

				Job j = new Job(
					new TestForPrime(test, index.multiply(N_BY),
						range.intValue()
					), "isprime-"+index, this, 60000
				);
				
				newJobs.add(j);
				jz.put("isprime-"+index, j);
			}

			cm.scheduleJobs( (Job[])newJobs.toArray(new Job[]{}) );
			newJobs.clear();
		} catch (Exception e) {
			e.printStackTrace();
			System.exit(0);
		}

		System.out.println("Now waiting for the answers.");

		//wait until we get a divisor or there are no jobs left.
		while(hasDivisor == false && jz.isEmpty() == false) {
			try {
				synchronized (jz) {
					jz.wait();
				}
				//Thread.sleep(1000);
			} catch (Exception e) {e.printStackTrace();}
		}

		//now kill all the remaining jobs.
		if(jz.size() != 0) {
			synchronized (jz) {
				Set vals = jz.entrySet();
				Iterator i = vals.iterator();
				while(i.hasNext()) {
					Map.Entry me = (Map.Entry)i.next();
					Job j = (Job)me.getValue();
					newJobs.add(j.jobID);
				}

				try {
					cm.abortJobs( (String[])newJobs.toArray(new String[]{}) );
					newJobs.clear();
				} catch (Exception e) {
					e.printStackTrace();
					System.exit(0);
				}
			}
		}
		System.out.println("Test took "+(System.currentTimeMillis() - start)/1000+" seconds to complete.");

		//display results.
		if(hasDivisor == false) {
			System.out.println(test+" is prime!");
		} else {
			System.out.println(test+" is composite.");
		}
		System.exit(0);
	}

	private boolean hasDivisor;
	private long last_update;
	private BigInteger biggest_seen;

	public void jobFinished(String jobID, Object result) {
		synchronized (jz) {
			Job j = (Job)jz.get(jobID);
			if(j == null) {System.out.println("bogus finished job: "+jobID); return;}
			jz.remove(jobID);
			hasDivisor |= ((Boolean)result).booleanValue();
			jz.notify();
		}

		String x = jobID.substring(8);

		BigInteger top = new BigInteger(x).multiply(N_BY);
		if(top.compareTo(biggest_seen) > 0) {
			biggest_seen = top;
		}
		if(System.currentTimeMillis() - last_update > 1000) {
			System.out.println("Largest potential divisor examined: "+biggest_seen);
			last_update = System.currentTimeMillis();
		}
	}

	public boolean shouldProceed() {
		//FIXME: not impl
		return true;
	}

	public void abortTask() {
		//FIXME: not impl
	}
}

// XXX: The job should be in a different file!

class TestForPrime implements Workable {
	private BigInteger test, start;
	private int howmuch;
	private static final BigInteger N0 = new BigInteger("0");
	private static final BigInteger N1 = new BigInteger("1");
	private static final BigInteger N2 = new BigInteger("2");

	public TestForPrime(BigInteger test, BigInteger start, int howmuch) {
		this.test = test;
		this.start = start;
		this.howmuch = howmuch;
	}

	public Object execute(TaskWorker control) {
		if(start.compareTo(N0) == 0) {
			//if the first number is 0, advance to 2
			start = N2;
			howmuch -= 2;
		} else if(start.compareTo(N1) == 0) {
			//if the first number is 1, advance to 2
			start = N2;
			howmuch -= 1;
		}
		BigInteger mod;

		//check for 2.
		if(start.equals(N2)) {
			mod = test.mod(start);
			if(mod.compareTo(N0) == 0) {
				System.err.println(start+" divides the number.");
				return Boolean.TRUE;
			}
			start = start.add(N1);
			howmuch--;
		}

		mod = start.mod(N2);
		if(mod.compareTo(N0) == 0) {
			System.out.println("Ack!  Testing with even!  Add one...");
			//even!  add one.
			start = start.add(N1);
		}

		for(int i = 0; i < howmuch; i++) {
			mod = test.mod(start);
			//System.out.print(test+" mod "+start+" = "+mod+"\r");
			if(mod.compareTo(N0) == 0) {
				System.err.println(start+" divides the number!");
				//no remainder!  that means that we've found a divisor.
				//so we can return true.
				return Boolean.TRUE;
			}
			start = start.add(N2);
			if(control.shouldProceed() == false) return null;
		}
		return Boolean.FALSE;

	}
}
