/* mutate various blocks in a file */
#define _ISOC99_SOURCE

#include <stdio.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <stdint.h>
#include <fcntl.h>
#include <string.h>
#include <strings.h>
#include <stdlib.h>
#include <errno.h>

#define STR_SIZE 512

#define ACTION_UNKNOWN -1
#define ACTION_ZERO    0
#define ACTION_REVERT  1
#define ACTION_BITFLIP 2
#define ACTION_GARBAGE 3
#define ACTION_ONE     4

/* Seed the random number generator */
static int seed_rng(void)
{
	unsigned int x;
	FILE *fp;

	fp = fopen("/dev/urandom", "r");
	if (!fp) {
		perror("/dev/urandom");
		return 0;
	}

	if (fread(&x, sizeof(int), 1, fp) != 1) {
		perror("/dev/urandom");
		return 0;
	}

	fclose(fp);

	srand(x);

	return 1;
}

/* Generate a random byte */
static unsigned char rand_byte(void)
{
	 return (unsigned char)(256.0 * (rand() / (RAND_MAX + 1.0)));
}

static void zero(char *buf, size_t len)
{
	memset(buf, 0, len);
}

static void one(char *buf, size_t len)
{
	memset(buf, 0xFF, len);
}

static void garbage(char *buf, size_t len)
{
	size_t i;

	for (i = 0; i < len; i++)
		buf[i] = rand_byte();
}

static void revert(char *buf, size_t len)
{
	/* empty */
}

static uint32_t rand_int(uint32_t max)
{
	uint32_t tmp;

	tmp = ((uint32_t)rand_byte() << 0)  | ((uint32_t)rand_byte() << 8)  |
	      ((uint32_t)rand_byte() << 16) | ((uint32_t)rand_byte() << 24);

	return (uint32_t) ( ((float)max) * (tmp / (UINT32_MAX + 1.0)));
}

static void bitflip(char *buf, size_t len)
{
	uint32_t bitno;

	bitno = rand_int(len * 8);
	buf[bitno / 8] = (buf[bitno / 8] & ~(2 << (bitno % 8))) | (~buf[bitno / 8] & (2 << (bitno % 8)));
}

struct {
	void (*action)(char *buf, size_t len);
} actions[] = {
	[ACTION_ZERO] = {zero},
	[ACTION_REVERT] = {revert},
	[ACTION_BITFLIP] = {bitflip},
	[ACTION_GARBAGE] = {garbage},
	[ACTION_ONE] = {one},
};

static int get_action(const char *str)
{
	if (!strcasecmp(str, "zero"))
		return ACTION_ZERO;
	else if (!strcasecmp(str, "revert"))
		return ACTION_REVERT;
	else if (!strcasecmp(str, "bitflip"))
		return ACTION_BITFLIP;
	else if (!strcasecmp(str, "garbage"))
		return ACTION_GARBAGE;
	else if (!strcasecmp(str, "one"))
		return ACTION_ONE;

	return ACTION_UNKNOWN;
}

int main(int argc, char *argv[])
{
	float probability;
	size_t block_size, block_num;
	char str[STR_SIZE];
	char *buf;
	int action;
	int in_fd, out_fd;
	unsigned char x;

	if (argc < 5) {
		printf("Usage: %s blocksize file probability corruption\n", argv[0]);
		return 0;
	}

	block_size = strtoul(argv[1], NULL, 0);
	if (errno) {
		perror(argv[1]);
		return 1;
	}

	out_fd = open(argv[2], O_RDWR | O_LARGEFILE);
	if (!out_fd) {
		perror(argv[2]);
		return 2;
	}

	probability = strtof(argv[3], NULL);
	if (errno) {
		perror(argv[3]);
		close(out_fd);
		return 3;
	}

	action = get_action(argv[4]);
	if (action == ACTION_UNKNOWN) {
		fprintf(stderr, "%s: Unknown action.\n", argv[4]);
		return 5;
	} else if (action == ACTION_REVERT) {
		if (argc != 6) {
			printf("Usage: %s blocksize file probability corruption old_file\n", argv[0]);
			return 6;
		}
		in_fd = open(argv[5], O_RDWR | O_LARGEFILE);
		if (!in_fd) {
			perror(argv[5]);
			close(in_fd);
			return 4;
		}
	} else {
		in_fd = out_fd;
	}

	buf = malloc(block_size);
	if (!buf) {
		perror("malloc");
		close(in_fd);
		return 3;
	}

	seed_rng();

	while (fgets(str, STR_SIZE, stdin)) {
		x = rand_byte();
		if ( (((float)x) / 255) > probability) {
			printf(".");
			fflush(stdout);
			continue;
		}

		printf("C");
		fflush(stdout);

		block_num = strtoul(str, NULL, 0);

		if (lseek64(in_fd, block_num * block_size, SEEK_SET) != block_num * block_size) {
			perror(argv[2]);
			continue;
		}

		if (read(in_fd, buf, block_size) != block_size) {
			perror(argv[2]);
			continue;
		}

		actions[action].action(buf, block_size);

		if (lseek64(out_fd, block_num * block_size, SEEK_SET) != block_num * block_size) {
			perror(argv[2]);
			continue;
		}

		if (write(out_fd, buf, block_size) != block_size) {
			perror(argv[2]);
			continue;
		}
	}
	printf("\n");

	free(buf);
	if (out_fd != in_fd)
		close(out_fd);
	close(in_fd);
	return 0;
}
